/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
Generate data to feed into the matlab programs 
5.7.2019 (revised 5.8.2020)
Pascual Restrepo
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

global project  "C:\Users\Pascual Restrepo\Dropbox\Research\wealth_inequality\AutomationInequality\replication_mrr_may_2020\s2a_calibration_model"

* calibration targets *
global capital_output_ratio=3
global cost_saving_gains=0.30

* BLS targets: from https://www.bls.gov/lpc/ *
* nonfinancial corporate sector 0.655 *
global bls_target_1980=0.655
global bls_target_2014=0.572
global beta_base=${bls_target_1980}

************************************************
****1: Compute alphas to match BLS targets  ****
************************************************
use  	"${project}/output/OMEGA_2000.dta", clear
gen 	omega_onet_base=omega_onet /*as discussed in the paper, we use exposures to routine jobs in 2000 to create the measures of change in alpha*/
keep 	qinc omega_onet_base

* construct measures of alpha as described in the appendix *
foreach year in 1980 2014{
merge 1:1 qinc using "${project}/output/WAGEDIST_`year'.dta", assert(3) nogenerate
rename ipums_*wage *wage`year'

* adjustment factor: adjusts calculation to account for changes in omega over time and deviations introduced by smoothing procedure *
egen adjustment_onet=wtmean(omega_onet_base), weight(hrwage`year') 

* calibrated to match labor share *
gen agg_invlabsh`year'=(1/${bls_target_`year'}-1/${beta_base})/adjustment_onet+1/${beta_base}

* inverse labor intensity by percentile *
gen inv_one_minus_alpha_`year'=omega_onet_base*(agg_invlabsh`year'-1/${beta_base})+1/${beta_base}

* alphas by percentile *
gen alpha`year'=1-1/inv_one_minus_alpha_`year'

drop adjustment_onet
}

keep qinc hrwage* inv_one_minus_alpha_* alpha* omega_* 
sort qinc
tempfile alpha_schedules
save `alpha_schedules', replace

** *******************************
**2: Obtain time series objects **
**********************************
use if year==1980 | year==2014 using "${project}/raw_data/aggregates.dta", clear /*loads data from FRED, BLS, and Fernald*/

* generate real gdp per worker *
gen gdp_real=.

* adjust for PCE price index (from BEA) *
replace gdp_real=gdp*(97.102/39.531) if year==1980
replace gdp_real=gdp*(97.102/109.157) if year==2014

* adjust for aggregate employment (civilian employment from BLS), and report in terms of hours, assuming 1,800 hours per worker per year *
replace gdp_real=1000000*gdp_real/civ_emp
sum gdp_real if year==1980
scalar Y_1980=r(mean)/1800
sum gdp_real if year==2014
scalar Y_2014=r(mean)/1800

* TFP statistics that will be used for comparisson *
sum tfp_fernald if year==1980
scalar TFP_1980=r(mean)
sum tfp_fernald if year==2014
scalar TFP_2014=r(mean)
scalar TFP_fernald_growth=ln(TFP_2014)-ln(TFP_1980)

*******************
**3: Calibration **
*******************
use `alpha_schedules', clear

* define base alpha and final alpha *
gen alpha_base  =alpha1980
gen alpha_final =alpha2014
gen change_alpha=alpha_final-alpha_base

* compute etas (note: only eta 1980 used in numerical exercise) *
gen eta_base =hrwage1980/(1-alpha_base)
gen eta_final=hrwage2014/(1-alpha_final)

* normalize etas so that they add up to 1 *
egen eta_tot_base=total(eta_base)
replace eta_base=eta_base/eta_tot_base
sort qinc

egen eta_tot_final=total(eta_final)
replace eta_final=eta_final/eta_tot_final
sort qinc

* verify calibration hits the targets up to numerical errors *
sum alpha_base [w=eta_base]
assert abs(r(mean)==1-${bls_target_1980})<0.0001

sum alpha_final [w=eta_final]
assert abs(r(mean)==1-${bls_target_2014})<0.0001

* baseline wages (consistent with national accounts---includes other compensation) *
gen w_base=100*eta_base*(1-alpha_base)*Y_1980

* capital-output ratio and rental rate of capital *
scalar KtoY_1980=${capital_output_ratio}
scalar R_1980=(1-${bls_target_1980})/KtoY_1980

* labor productivity relative to capital by wage percentile *
gen psi_z=(w_base/R_1980)/(1+${cost_saving_gains})

* compute productivity gains from automation (reported in the paper) *
egen TFP_gains_autom=total(eta_base*max(ln(w_base/(psi_z*R_1980)),0)*max(alpha_final-alpha_base,0))
sum TFP_gains_autom
display 100*r(mean) /*gains from automation*/
display 100*TFP_fernald_growth /*total tfp increase during this period*/

* compute scale parameter A to match aggregates per hour *
gen prod_p=eta_base*(${bls_target_1980})*ln((1/100)*psi_z/(eta_base*(${bls_target_1980})))
egen prod=total(prod_p)
sum prod
scalar A_1980=Y_1980^(${bls_target_1980})*(KtoY_1980/(1-${bls_target_1980}))^(-(1-${bls_target_1980}))*exp(-r(mean))

***************************************
**3.7 Export data to be used in matlab**
***************************************
gen R_target=R_1980
gen A_target=A_1980
assert psi_z!=.
assert eta_base!=.
assert eta_final!=.
assert alpha_base!=.
assert alpha_final!=.
export delimited qinc eta_base alpha_base eta_final alpha_final R_target A_target psi_z using ///
       "${project}/output/data_calibration_matlab_pi30.csv", replace
	      